/*
 * Copyright (c) 1996 Picture Elements, Inc.
 *    Stephen Williams (steve@icarus.com)
 *
 *    c/o Picture Elements, Inc.
 *    777 Panoramic Way
 *    Berkeley, CA, 94704
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Library General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Library General Public License for more details.
 *
 *    You should have received a copy of the GNU Library General Public
 *    License along with this library; if not, write to the Free
 *    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 *    MA 02111-1307, USA
 */
#ident "$Id: pci_init.c,v 1.2 1997/05/21 03:04:32 steve Exp $"
/*
 * $Log: pci_init.c,v $
 * Revision 1.2  1997/05/21 03:04:32  steve
 *  Add support for Windows NT.
 *
 * Revision 1.1  1997/05/20 23:57:08  steve
 *  Make the extension a shared object
 *  Make filenames more 8.3 friendly
 *
 * Revision 1.3  1996/10/10 01:17:25  steve
 *  Fix a bogus truncation working with dwords.
 *
 * Revision 1.2  1996/10/01 00:47:10  steve
 *  Copy notices.
 *
 * Revision 1.1.1.1  1996/09/23 20:20:22  steve
 *  Initial release.
 *
 */

/*
 * This extension allows TCL to use the pciconf driver to read/write
 * configuration space of PCI devices.
 *
 * pciopen <name> [-r] [-w] <path>
 *
 *	Use this to open the device and create an object <name> that
 *	can receive all the subcommands related to manipulating the device.
 *
 * <name> select <vendor> <device> <instance>
 *
 *	Use this to select a device to manipulate. If the device
 *	exists, this succeeds and get and set start applying to that
 *	board. If the device does not exist, this returns an error.
 *
 * <name> get_byte <offset>
 * <name> get_word <offset>
 * <name> get_dword <offset>
 *
 *	Get a byte/word/dword from the specific location. Remember
 *	that the PCI configuration space is defined to be little-
 *	endian. The offset must be properly aligned for the access.
 *
 *	The result is the integer value stored at that location, in
 *	decimal.
 *
 * <name> set_byte <offset> <value>
 * <name> set_word <offset> <value>
 * <name> set_dword <offset> <value>
 *
 *	Set the byte/word/dword at the specific location. As with the
 *	get operations, the offset must be properly aligned for the write.
 */

# include  <tcl.h>
# include  <fcntl.h>
# include  "pciconf.h"
#ifdef WINNT
# include  <windows.h>
# include  <io.h>
typedef long off_t;
#endif

struct PciconfInst {
#ifdef WINNT
      HANDLE fd;
#else
      int fd;
#endif
};

static int PciconfCmd(ClientData cd, Tcl_Interp *interp, int argc,
		      char*argv[])
{
      struct PciconfInst *inst = (struct PciconfInst*)cd;

      if (argc == 1) {
	    Tcl_SetResult(interp, "missing subcommand", TCL_STATIC);
	    return TCL_ERROR;
      }

      if (strcmp(argv[1],"get_byte") == 0) {
	    unsigned char val = 0;
	    off_t off;
	    int rc;
	    char buf[6];

	    if (argc != 3) {
		  Tcl_SetResult(interp,
				"Usage: get_byte <offset>",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

	    off = strtoul(argv[2], 0, 0);
#if defined(WINNT)
	    rc = SetFilePointer(inst->fd, off, 0, FILE_BEGIN);
	    ReadFile(inst->fd, &val, 1, &rc, 0);
#else
	    rc = lseek(inst->fd, off, SEEK_SET);
	    rc = read(inst->fd, &val, 1);
#endif
	    sprintf(buf, "%u", val);
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);

     } else if (strcmp(argv[1],"get_dword") == 0) {
	    unsigned long val = 0;
	    off_t off;
	    int rc;
	    unsigned char buf[4];
	    char str[16];

	    if (argc != 3) {
		  Tcl_SetResult(interp,
				"Usage: get_dword <offset>",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

	    off = strtoul(argv[2], 0, 0);
	    if ((off%4) != 0) {
		  Tcl_SetResult(interp, "get_dword: offset not aligned.",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

#if defined(WINNT)
	    rc = SetFilePointer(inst->fd, off, 0, FILE_BEGIN);
	    ReadFile(inst->fd, buf, 4, &rc, 0);
#else
	    rc = lseek(inst->fd, off, SEEK_SET);
	    rc = read(inst->fd, buf, 4);
#endif

	    val = buf[0] + (buf[1]<<8) + (buf[2]<<16) + (buf[3]<<24);
	    sprintf(str, "%lu", val);
	    Tcl_SetResult(interp, str, TCL_VOLATILE);

     } else if (strcmp(argv[1],"get_word") == 0) {
	    unsigned short val;
	    off_t off;
	    int rc;
	    unsigned char buf[2];
	    char str[6];

	    if (argc != 3) {
		  Tcl_SetResult(interp,
				"Usage: get_word <offset>",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

	    off = strtoul(argv[2], 0, 0);
	    if ((off%2) != 0) {
		  Tcl_SetResult(interp, "get_word: offset not aligned.",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

#if defined(WINNT)
	    rc = SetFilePointer(inst->fd, off, 0, FILE_BEGIN);
	    ReadFile(inst->fd, buf, 2, &rc, 0);
#else
	    rc = lseek(inst->fd, off, SEEK_SET);
	    rc = read(inst->fd, buf, 2);
#endif

	    val = buf[0] + (buf[1]<<8);
	    sprintf(str, "%hu", val);
	    Tcl_SetResult(interp, str, TCL_VOLATILE);

      } else if (strcmp(argv[1],"set_byte") == 0) {
	    unsigned char val;
	    off_t off;
	    int rc;

	    if (argc != 4) {
		  Tcl_SetResult(interp,
				"Usage: set_byte <offset> <value>",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

	    off = strtoul(argv[2], 0, 0);
	    val = strtoul(argv[3], 0, 0);

#if defined(WINNT)
	    rc = SetFilePointer(inst->fd, off, 0, FILE_BEGIN);
	    WriteFile(inst->fd, &val, 1, &rc, 0);
#else
	    rc = lseek(inst->fd, off, SEEK_SET);
	    rc = write(inst->fd, &val, 1);
#endif

      } else if (strcmp(argv[1],"set_dword") == 0) {
	    unsigned long val;
	    unsigned char buf[4];
	    off_t off;
	    int rc;

	    if (argc != 4) {
		  Tcl_SetResult(interp,
				"Usage: set_dword <offset> <value>",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

	    off = strtoul(argv[2], 0, 0);
	    if ((off%4) != 0) {
		  Tcl_SetResult(interp, "set_dword: offset not aligned.",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

	    val = strtoul(argv[3], 0, 0);
	    buf[0] = val % 256; val /= 256;
	    buf[1] = val % 256; val /= 256;
	    buf[2] = val % 256; val /= 256;
	    buf[3] = val;

#if defined(WINNT)
	    rc = SetFilePointer(inst->fd, off, 0, FILE_BEGIN);
	    WriteFile(inst->fd, buf, 4, &rc, 0);
#else
	    rc = lseek(inst->fd, off, SEEK_SET);
	    rc = write(inst->fd, buf, 4);
#endif

      } else if (strcmp(argv[1],"set_word") == 0) {
	    unsigned short val;
	    unsigned char buf[2];
	    off_t off;
	    int rc;

	    if (argc != 4) {
		  Tcl_SetResult(interp,
				"Usage: set_word <offset> <value>",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

	    off = strtoul(argv[2], 0, 0);
	    if ((off%2) != 0) {
		  Tcl_SetResult(interp, "set_word: offset not aligned.",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

	    val = strtoul(argv[3], 0, 0);
	    buf[0] = val % 256; val /= 256;
	    buf[1] = val;

#if defined(WINNT)
	    rc = SetFilePointer(inst->fd, off, 0, FILE_BEGIN);
	    WriteFile(inst->fd, buf, 2, &rc, 0);
#else
	    rc = lseek(inst->fd, off, SEEK_SET);
	    rc = write(inst->fd, buf, 2);
#endif

      } else if (strcmp(argv[1],"select") == 0) {
	    struct pciconf_info info;
	    int rc;

	    if (argc < 5) {
		  Tcl_SetResult(interp,
				"usage: select <vendor> <device> <inst>",
				TCL_STATIC);
		  return TCL_ERROR;
	    }

	    info.vendor_id = strtoul(argv[2], 0, 0);
	    info.device_id = strtoul(argv[3], 0, 0);
	    info.instance  = strtoul(argv[4], 0, 0);

#if defined(WINNT)
	    {
		  BOOL flag;
		  long ret;
		  flag = DeviceIoControl(inst->fd, PCICONF_SELECT,
					 &info, sizeof info, 0, 0,
					 &ret, 0);
		  if (!flag) rc = -1;
		  else rc = 0;
	    }
#else
	    rc = ioctl(inst->fd, PCICONF_SELECT, &info);
#endif
	    if (rc == -1) {
		  Tcl_SetResult(interp, "select failed", TCL_STATIC);
		  return TCL_ERROR;
	    }

      } else {
	    Tcl_AppendResult(interp, "invalid subcommand: ", argv[1], 0);
	    return TCL_ERROR;
      }

      return TCL_OK;
}

static void PciconfDeleteCmd(ClientData data)
{
      struct PciconfInst *inst = (struct PciconfInst*)data;

#if defined(WINNT)
      CloseHandle(inst->fd);
#else
      close(inst->fd);
#endif
      free(inst);
}

static int PciOpen(ClientData cd, Tcl_Interp *interp, int argc,
		     char*argv[])
{
#if defined(WINNT)
      HANDLE fd;
      DWORD o_flag = 0;
#else
      int fd;
      int o_flag = 0;
#endif
      struct PciconfInst*inst;

      const char*path = 0;
      int r_flag = 0, w_flag = 0;
      unsigned idx;

      if (argc < 3) {
	    Tcl_SetResult(interp, "wrong # args", TCL_STATIC);
	    return TCL_ERROR;
      }

      for (idx = 2 ;  (idx < argc) && (path == 0) ;  idx += 1) {

	    if (strcmp(argv[idx],"-r") == 0) {
		  r_flag = 1;
	    } else if (strcmp(argv[idx],"-w") == 0) {
		  w_flag = 1;
	    } else {
		  path = argv[idx];
	    }

      }

#if defined(WINNT)
      if (r_flag) o_flag |= GENERIC_READ;
      if (w_flag) o_flag |= GENERIC_WRITE;

      fd = CreateFile(path, o_flag, 0, 0, OPEN_EXISTING,
		      FILE_ATTRIBUTE_NORMAL, 0);
      if (fd == INVALID_HANDLE_VALUE) {
	    Tcl_AppendResult(interp, "error opening ``", path, "''", 0);
	    return TCL_ERROR;
      }
#else
      switch (r_flag + 2*w_flag) {
	  case 0:
	    Tcl_SetResult(interp, "missing -r or -w flag",
			  TCL_STATIC);
	    return TCL_ERROR;

	  case 1:
	    o_flag = O_RDONLY;
	    break;

	  case 2:
	    o_flag = O_WRONLY;
	    break;

	  case 3:
	    o_flag = O_RDWR;
	    break;
      }

      fd = open(path, o_flag, 0);
      if (fd == -1) {
	    Tcl_AppendResult(interp, "error opening ``", path, "''", 0);
	    return TCL_ERROR;
      }
#endif

      inst = (struct PciconfInst*)malloc(sizeof(struct PciconfInst));
      inst->fd = fd;

      Tcl_CreateCommand(interp, argv[1], PciconfCmd, inst, PciconfDeleteCmd);
      return TCL_OK;
}


int Pciconf_Init(Tcl_Interp *interp)
{
      Tcl_CreateCommand(interp, "pciopen", PciOpen, 0, 0);
      return TCL_OK;
}
