#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pcimem.h>
#include "memmap.h"
#include "phibutil.h"

#ifndef DEBUG
#define DEBUG (0)
#endif
#if DEBUG
#define Cfprintf fprintf
#define Cprintf printf
#else
#define Cfprintf
#define Cprintf
#endif

/* PHIB specific constants */
#define RAMSIZE (PHIBRAMWORDS)
#define RAMADDR (0x00000000)
#define LBCADDR (1<<(15-2))
#define CRST (1<<(7-2)) /* lai[CRST] read causes CB reset */
#define LRST (1<<(8-2)) /* lai[LRST] read causes HIB local reset */
#define CLR (1<<(9-2)) /* lai[CLR] read clears address counters */
#define DWREQ (1<<(6-2)) /* lai[DWREQ] read issues DMA write request */
#define QUERY_DMAW (1<<(10-2)) /* la[ISDMAW] read returns /dma_w status to ldsts */
#define QUERY_DMAR (1<<(11-2)) /* la[ISDMAR] read returns /dma_r status to ldsts */
#define QUERY_CALCSTS (0) /* la[ISDMAR] read returns /calcsts status to ldsts */
#define STSBIT (1<<15)

#ifdef THREAD_SAFE_PHIBLIB
#include  <pthread.h>
pthread_mutex_t phibmutex = PTHREAD_MUTEX_INITIALIZER;
#endif /* THREAD_SAFE_PHIBLIB */
static int mutexlock(void);
static int mutexunlock(void);

/* PHIB local function(s) */
static int read_dualportram_timeout(int devid, int size);
static void start_read_dualportram(int devid, int size, unsigned int *buf, double ratio);
static void danger_read_dualportram(int devid, int size, double ratio);
static void read_dualportram(int devid, int size);
static void read_dualportramDMA(int devid, int size);
static void read_dualportramPIO(int devid, int size);
static void clear_cbcounter(int devid, int checkmax);
static int read_cbstatus(int devid, unsigned int query);
static int read_cbcounter(int devid, int checkmax);
static void wait_dmar_assertion(int devid);
static void wait_dmar_deassertion(int devid);
static void testrun(int devid);
static void waitfor(double maxcount);

/* PHIB local variable(s) */
static unsigned int off[NPCIMEM];
static unsigned int *resbufp[NPCIMEM];
static unsigned long resbufpa[NPCIMEM]; /* physical addres for resbufp[] */
static unsigned int *sharedbufp;
#ifdef __linux__
static unsigned long sharedbufpa; /* physical addres for sharedbufp */
#else
static unsigned long sharedbufpa[NPCIMEM]; /* physical addres for sharedbufp */
#endif

static unsigned int * phib[NPCIMEM];


#if defined(__alpha) || defined(__DECC)

/* optimized for:
 * VT-Alpha 533MHz/256MB, PHIB/9080/32MHz, GRAPE-5/4GCs/15MHz */
static double dmawwait = 0.0;
static double dmarwait = 0.0;

#elif defined(__linux__)

/*
static double dmawwait = 0.4;
static double dmarwait = 0.65;
*/

static double dmawwait = 0.4;
static double dmarwait = 1.0;

#else
static double dmawwait = 2.0;
static double dmarwait = 3.0;
#endif


static int nclusters = 1;
static int firstcluster = 0;

/*
 * alloc phib[cluster0] to phib[nclusters]
 */
void
phib_alloc(int phib0, int nphibs)
{
    int i;

    if (nphibs <= 0) {
	fprintf(stderr, "invalid # of PHIB (%d). abort.\n",
		nphibs);
	exit(1);
    }
    if (phib0 < 0) {
	fprintf(stderr, "invalid PHIB ID (%d). abort.\n",
		phib0);
	exit(1);
    }
    if (phib0 + nphibs > NPCIMEM) {
	fprintf(stderr, "too large PHIB ID (=%d). abort.\n",
		nphibs);
	exit(1);
    }
    nclusters = nphibs;
    firstcluster = phib0;
    fprintf(stderr, "# of clusters: %d  (allocated phib[", nclusters);
    for (i = phib0; i < phib0+nphibs; i++) {
	fprintf(stderr, "%d", i);
	if (i < phib0+nphibs-1) {
	    fprintf(stderr, ", ");
	}
    }
    fprintf(stderr, "])\n");
}

void
phib_set_nclusters(int n)
{
    if (0 < n && n <= NPCIMEM)
    {
	nclusters = n;
	fprintf(stderr, "# of clusters: %d\n", nclusters);
    }
    else
    {
	nclusters = 1;
	fprintf(stderr,
		"inappropriate # of clusters (=%d)\nuse default (=1).",
		n);
    }
}

int
phib_get_nclusters(void)
{
    return (nclusters);
}

void
phib_optimizeDMA(double ww, double rw)
{
    dmawwait = ww;
    dmarwait = rw;
}

/* interfaces for single cluster */
unsigned int *
phib_open(void)
{
    return (phib_openMC(firstcluster));
}

void
phib_close(void)
{
    phib_closeMC(firstcluster);
}

void
phib_piow(unsigned int addr, unsigned int cmd)
{
    phib_piowMC(firstcluster, addr, cmd);
}

void
phib_dmar(int size, unsigned int *buf)
{
    phib_dmarMC(firstcluster, size, buf);
}

void
phib_dmaw(int size, unsigned int *buf)
{
    phib_dmawMC(firstcluster, size, buf);
}

int
phib_dmaw_timeout(int size, unsigned int *buf)
{
    return(phib_dmaw_timeoutMC(firstcluster, size, buf));
}

int
phib_calcsts(void)
{
    return (phib_calcstsMC(firstcluster));
}

void
phib_wait(int nclk)
{
    phib_waitMC(firstcluster, nclk);
}

void
phib_pioconfw(unsigned int addr, unsigned int cmd)
{
    phib_pioconfwMC(firstcluster, addr, cmd);
}

unsigned int *
phib_mapped_addrMC(int devid)
{
    return (phib[devid]);
}

/*
 * open PHIB
 * returns pointer to the data buffer
 */
unsigned int *
phib_openMC(int devid)
{
    unsigned int *ret;

    ret = phib_open_notestMC(devid);
    testrun(devid);
    return (ret);
}

#ifdef __linux__

unsigned int *
phib_open_notestMC(int devid)
{
    int ic, ic0;
    unsigned int val;
    static int firstcall = 1;
    int psz = getpagesize();

    if (nclusters+firstcluster <= devid)
    {
	fprintf(stderr,
		"phib_openMC(): too large devid(= %d).\n",
		devid);
	exit(1);
    }
    if (firstcluster > devid)
    {
	fprintf(stderr,
		"phib_openMC(): too small devid(= %d).\n",
		devid);
	exit(1);
    }
    while (-1 == (long int)(phib[devid] = TBopen(devid)))
    {
	sleep(1);
    }
    mutexlock();
    if (firstcall)
    {
	firstcall = 0;
	/*
	 * resbufp[devid]: virtual address of DMA buffer
	 * resbufpa[devid]: physical address of resbufp[devid]
	 * sharedbufp: alias to resbufp[firstcluster]
	 * sharedpa: physical address of shared bufp
	 *
	 */
	TBgetDmaInfo(devid, &(resbufpa[devid]), &(resbufp[devid]));
	if (firstcluster == devid)
	{
	    sharedbufpa = resbufpa[devid];
	}
    }
    mutexunlock();
    Cfprintf(stderr, "phib%d: resbufp: 0x%016lx resbufpa %016lx\n",
	    devid, resbufp[devid], resbufpa[devid]);
    val = TBregRead(devid, 0x6c);
    TBregWrite(devid, 0x6c, (val & 0xfffffff0)|0xc); /* mem read multiple */
    TBregWrite(devid, 0x84, resbufpa[devid]);

    return (unsigned int *)resbufp[devid];
}

#else  /* tru64 */

unsigned int *
phib_open_notestMC(int devid)
{
    int ic, ic0;
    unsigned int val;
    static int firstcall = 1;
    int psz = getpagesize();

    Cfprintf(stderr, "phib_open_notestMC: will TBopen\n");

    if (nclusters+firstcluster <= devid)
    {
	fprintf(stderr,
		"phib_openMC(): too large devid(= %d).\n",
		devid);
	exit(1);
    }
    if (firstcluster > devid)
    {
	fprintf(stderr,
		"phib_openMC(): too small devid(= %d).\n",
		devid);
	exit(1);
    }

    while (-1 == (int)(phib[devid] = TBopen(devid)))
    {
	sleep(1);
    }
    Cfprintf(stderr, "phib_open_notestMC: TBopen done\n");
    mutexlock();
    if (firstcall)
    {
	firstcall = 0;

	/*
	 * resbufp[devid]: virtual address of DMA buffer
	 * resbufpa[devid]: physical address of resbufp[devid]
	 * sharedbufp: alias to resbufp[firstcluster]
	 * sharedpa: physical address of shared bufp
	 *
	 */

	/* allocate all necessary buffer in a single calloc() call */
	sharedbufp = 
	    (unsigned int *)calloc(sizeof(unsigned int), RAMSIZE*(NPCIMEM+1));
	if (sharedbufp == NULL)
	{
	    fprintf(stderr, "phib_openMC(): calloc() failed\n");
	    exit (2);
	}
	/* align to the page boundary */
	if ((unsigned long)sharedbufp%psz)
	{
	  sharedbufp = (void *)(((unsigned long)sharedbufp/psz+1)*psz);
	}
	/* set resbufp[devid] */
	for (ic = firstcluster, ic0 = 0; ic < nclusters+firstcluster; ic++, ic0++)
	{
	    off[ic] = 0;
	    resbufp[ic] = sharedbufp + RAMSIZE*(ic0);
	}
    }
    mutexunlock();

    /*
     * note that TBdmaMapLoad must be invoked everytime after another process invoked it
     */
    resbufpa[devid] = TBdmaMapLoad(devid, (unsigned int *)(resbufp[devid]), RAMSIZE);
    sharedbufpa[devid] = resbufpa[devid];

    val = TBregRead(devid, 0x6c);
    TBregWrite(devid, 0x6c, (val & 0xfffffff0)|0xc); /* mem read multiple */
    TBregWrite(devid, 0x84, resbufpa[devid]);

    return (unsigned int *)resbufp[devid];
}

#endif


/* close PHIB */
void
phib_closeMC(int devid)
{
    TBterm(devid);
}

/* wait for more than nclk*30ns */

#if 1

unsigned int g5dummy[1000];
void
phib_waitMC(int devid, int nclk)
{
    int j;
    int nmax = 1000;
    int dummyoff = 1000;

    while (nclk > 0)
    {
	if (nclk < nmax)
	{
	    nmax = nclk;
	}
	for (j = 0; j < nmax; j++)
	{
	  /*
	    *(((unsigned int *)phib[devid])+j+dummyoff) = 0;
	  g5dummy[j] = *(((unsigned int *)phib[devid])+j+dummyoff);
	    */
	    *(((unsigned int *)phib[devid])+j+dummyoff) = 0;
	}
	nclk -= nmax;
    }
}

#else

int g5dummy;
void
phib_waitMC(int devid, int nclk)
{
    int i, j, k;
    int nmax = 100;

    nclk = 200;

    for (j = 0; j < nclk; j++)
    {
	for (i = 0; i < nmax; i++)
	{
	    g5dummy++;
	}
    }
}

#endif

/* send 'cmd' to PG with PIO write operation */
void
phib_piowMC(int devid, unsigned int addr, unsigned int cmd)
{
    unsigned int * mp = NULL;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_piowMC: open phib%d first.\n", devid);
	exit (2);
    }

    clear_cbcounter(devid, 0);

    TBmemWriteInt(off[devid], addr);
    off[devid]++;
    TBmemWriteInt(off[devid], cmd);
    off[devid]++;
    TBmemWriteInt(LBCADDR, off[devid]);
}

/* wait until PIOW finishes */
void
phib_pioconfwMC(int devid, unsigned int addr, unsigned int cmd)
{
    unsigned int * mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_pioconfwMC: open phib%d first.\n", devid);
	exit (2);
    }

    clear_cbcounter(devid, 0);
    TBmemWriteInt(off[devid], addr);
    off[devid]++;
    TBmemWriteInt(off[devid], cmd);
    off[devid]++;
    TBmemWriteInt(LBCADDR, off[devid]);
    while (read_cbcounter(devid, 0) != 2)
    {
    }
}

void
phib_add_rcntMC(int devid, int cnt)
{
    unsigned int * mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_add_rcntMC: open phib%d first.\n", devid);
	exit (2);
    }

    off[devid] += cnt;

    *(((u_int *)mp)+(LBCADDR)) = off[devid];
/*
    TBmemWriteInt(LBCADDR, off[devid]);
    */
}

void
phib_wait_dmar_assertionMC(int devid)
{
    wait_dmar_assertion(devid);
}

void
phib_wait_dmar_deassertionMC(int devid)
{
    wait_dmar_deassertion(devid);
}

/* send 'size' words from 'buf' to PG
   with DMA read operation */

#if NODMA /* PIO write transfer */
void
phib_dmarMC(int devid, int size, unsigned int *buf)
{
    int i, oft;
    unsigned int * mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_dmarMC(): open phib%d first.\n", devid);
	exit(2);
    }

    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "phib_dmarMC(): offset+size too large.\n");
	fprintf(stderr, "off: %d  size: %d\n", off[devid], size);
	exit(2);
    }

    wait_dmar_assertion(devid);

    oft = off[devid];
    for (i = 0; i < size; i++) {
	mp[oft] = buf[i];
	oft++;
    }

    off[devid] = oft;
    TBmemWriteInt(LBCADDR, off[devid]);
    wait_dmar_deassertion(devid);
}

#else /* DMA read transfer */

void
phib_dmarMC(int devid, int size, unsigned int *buf)
{
    phib_dmar_nowaitMC(devid, size, buf);
    wait_dmar_deassertion(devid);
}

#endif /* NODMA */


#ifdef __linux__

void
phib_broadcast(int size, unsigned int *buf)
{
    int i, ic;
    unsigned int *mp = NULL;

    for (ic = firstcluster; ic < nclusters+firstcluster; ic++)
    {
	if (off[ic]+size > RAMSIZE)
	{
	    fprintf(stderr, "phib_dmarMC(): offset+size too large.\n");
	    fprintf(stderr, "off: %d  size: %d\n", off[ic], size);
	    exit(2);
	}
	if (off[ic] != off[firstcluster])
	{
	    fprintf(stderr, "off[%d]: %d not sync with off[firstcluster]: %d\n",
		    ic, off[ic], off[firstcluster]);
	    exit(2);
	}

	if ((unsigned int *)sharedbufp > buf ||
	    (unsigned int *)(sharedbufp)+RAMSIZE < buf+size)
	{
	    TBregWrite(ic, 0x84, sharedbufpa);
	    for (i = 0; i < size; i++)
	    {
		sharedbufp[i] = buf[i];
	    }

	    fprintf(stderr, "!!! sharedbufp: 0x%08lx  buf: 0x%08lx  off: %d, size: %d\n",
		    (long)sharedbufp, (long)buf, off[ic], size);
	}
	else
	{
	    /* start address in PCI space */
	    TBregWrite(ic, 0x84, (unsigned long)((unsigned int*)sharedbufpa+(buf - sharedbufp)));
	}

	TBregWrite(ic, 0x88, off[ic]*4); /* start address in local space */
	TBregWrite(ic, 0x8c, size*4); /* size in byte */
	TBregWrite(ic, 0x90, 0x00000000); /* direction (0:PCI --> local) */
    }

    for (ic = firstcluster; ic < nclusters+firstcluster; ic++)
    {
	wait_dmar_assertion(ic);
	TBregWrite(ic, 0xa8, 0x00000003); /* start DMA */

#define DMAPAR (0)

#if DMAPAR
    }
#endif

    waitfor(dmarwait*size);

#if DMAPAR
    for (ic = firstcluster; ic < nclusters+firstcluster; ic++)
    {
#endif
	mp = phib[ic];
	if (!mp)
	{
	    fprintf(stderr, "phib_broadcast(): open phib%d first.\n", ic);
	    exit(2);
	}
	while (!(TBregRead(ic, 0xa8) & 0x10))
	{
	}
	off[ic] += size;
	TBmemWriteInt(LBCADDR, off[ic]);
    }
}

void
phib_broadcast_raw(int size, unsigned int *buf)
{
    int i, ic;

    for (ic = firstcluster; ic < nclusters+firstcluster; ic++)
    {
	if (off[ic]+size > RAMSIZE)
	{
	    fprintf(stderr, "phib_dmarMC(): offset+size too large.\n");
	    fprintf(stderr, "off: %d  size: %d\n", off[ic], size);
	    exit(2);
	}
	if (off[ic] != off[firstcluster])
	{
	    fprintf(stderr, "off[%d]: %d not sync with off[firstcluster]: %d\n",
		    ic, off[ic], off[firstcluster]);
	    exit(2);
	}

	if (sharedbufp > buf ||
	    sharedbufp+RAMSIZE < buf+off[ic]+size)
	{
	    TBregWrite(ic, 0x84, sharedbufpa);
	    for (i = 0; i < size; i++)
	    {
		sharedbufp[i] = buf[i];
	    }
	}
	else
	{
	    /* start address in PCI space */
	    TBregWrite(ic, 0x84, (unsigned long)((unsigned int*)sharedbufpa+(buf - sharedbufp)));
	}

	TBregWrite(ic, 0x88, off[ic]*4); /* start address in local space */
	TBregWrite(ic, 0x8c, size*4); /* size in byte */
	TBregWrite(ic, 0x90, 0x00000000); /* direction (0:PCI --> local) */
    }

    for (ic = firstcluster; ic < nclusters+firstcluster; ic++)
    {
	TBregWrite(ic, 0xa8, 0x00000003); /* start DMA */

#define DMAPARRAW (0)

#if DMAPARRAW
    }
#endif

    waitfor(dmarwait*size);

#if DMAPARRAW
    for (ic = firstcluster; ic < nclusters+firstcluster; ic++)
    {
#endif
	while (!(TBregRead(ic, 0xa8) & 0x10))
	{
	}
    }
}

#else /* tru64 */

void
phib_broadcast(int size, unsigned int *buf)
{
  fprintf(stderr, "phib_broadcast() is not implemented on Tru64\n");
  exit(0);
}

void
phib_broadcast_raw(int size, unsigned int *buf)
{
  fprintf(stderr, "phib_broadcast_raw() is not implemented on Tru64\n");
  exit(0);
}

#endif


void
phib_dmar_nowaitMC(int devid, int size, unsigned int *buf)
{
    int i;
    unsigned int * mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_dmar_nowaitMC(): open phib%d first.\n", devid);
	exit(2);
    }

    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "phib_dmarMC(): offset+size too large.\n");
	fprintf(stderr, "off: %d  size: %d\n", off[devid], size);
	exit(2);
    }

    if ((unsigned int *)resbufp[devid] > buf ||
	(unsigned int *)(resbufp[devid])+RAMSIZE < buf+size)
    {
	TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */

	for (i = 0; i < size; i++)
	{
	    ((unsigned int *)(resbufp[devid]))[i] = buf[i];
	}
    }
    else
    {
	/* start address in PCI space */
	TBregWrite(devid, 0x84,
		   (unsigned long)((unsigned long *)resbufpa[devid]+
				  (buf - resbufp[devid])));
    }

    TBregWrite(devid, 0x88, off[devid]*4); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000000); /* direction (0:PCI --> local) */
    wait_dmar_assertion(devid);
    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */

    waitfor(dmarwait*size);

    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }

    off[devid] += size;
    TBmemWriteInt(LBCADDR, off[devid]);
}

void
phib_dmar_rawMC(int devid, int size, unsigned int *buf)
{
    int i;
    unsigned int * mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_dmar_rawMC(): open phib%d first.\n", devid);
	exit(2);
    }

    if (size > RAMSIZE)
    {
	fprintf(stderr, "phib_dmar_rawMC(): size too large.\n");
	fprintf(stderr, "size: %d\n", size);
	exit(2);
    }

    if ((unsigned int *)resbufp[devid] > buf ||
	(unsigned int *)(resbufp[devid])+RAMSIZE < buf+size)
    {
	TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */
	for (i = 0; i < size; i++)
	{
	    ((unsigned int *)(resbufp[devid]))[i] = buf[i];
	}
    }
    else
    {
	/* start address in PCI space */

	Cfprintf(stderr, "buf: %08x   resbufp[%d]: %08x\n",
		 buf, devid, resbufp[devid]);

	TBregWrite(devid, 0x84,
		   (unsigned long)((unsigned int *)resbufpa[devid]+
				   (buf - (unsigned int *)resbufp[devid])));
    }

    TBregWrite(devid, 0x88, 0); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000000); /* direction (0:PCI --> local) */
    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */

    waitfor(dmarwait*size);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
}

/* receive 'size' words from PG to 'buf'
   with DMA write operation */
void
phib_dmawMC(int devid, int size, unsigned int *buf)
{
    int i;

    read_dualportram(devid, size);

    /* copy contents of 'resbufp[devid]' to 'buf' buffer
       specified by the user of this library */
    if ((unsigned int *)resbufp[devid] != buf)
    {
	for (i = 0; i < size; i++)
	{
	    buf[i] = ((unsigned int *)resbufp[devid])[i];
	}
    }
}

int
phib_dmaw_timeoutMC(int devid, int size, unsigned int *buf)
{
    int ret;
    int i;

    ret = read_dualportram_timeout(devid, size);

    /* copy contents of 'resbufp[devid]' to 'buf' buffer
       specified by the user of this library */
    if ((unsigned int *)resbufp[devid] != buf)
    {
	for (i = 0; i < size; i++)
	{
	    buf[i] = ((unsigned int *)resbufp[devid])[i];
	}
    }
    return(ret);
}

void
phib_start_dmawMC(int devid, int size, unsigned int *buf, double ratio)
{
    start_read_dualportram(devid, size, buf, ratio);
}

void
phib_finish_dmawMC(int devid, int len)
{
    waitfor(dmawwait*len);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
}

void
phib_danger_dmawMC(int devid, int size, unsigned int *buf, double ratio)
{
    danger_read_dualportram(devid, size, ratio);
}

void
phib_waitfor(int size)
{
    waitfor(dmawwait*size);
}


int
phib_calcstsMC(int devid)
{
    unsigned int val;
	
    val = read_cbstatus(devid, LBCADDR|QUERY_CALCSTS);
    MB;

    Cfprintf(stderr, "val: %d\n", val);

    return (val);
}

/*
 * receive data from dual-port ram to 'resbufp[devid]'.
 */
#define NTRYMAX (10)

static void
start_read_dualportram(int devid, int size, unsigned int *buf, double ratio)
{
    register int rcnt = 0;
    register int dmastart;
	
    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "danger_read_dualportram(): offset+size too large.\n");
	exit(2);
    }

    if ((unsigned int *)resbufp[devid] > buf ||
	(unsigned int *)(resbufp[devid])+RAMSIZE < buf+size)
    {
	fprintf(stderr, "start_read_dualportram(): inappropriate DMAW range\n");
	fprintf(stderr, "off: %d  size: %d\n", off[devid], size);
	fprintf(stderr, "resbufp[%d]: %08lx buf: %08lx\n",
		devid, (long)resbufp[devid], (long)buf);
	exit(1);
    }
    else
    {
	/* start address in PCI space */
	TBregWrite(devid, 0x84,
		   (unsigned long)((unsigned int *)resbufpa[devid]+
				  (buf - (unsigned int *)resbufp[devid])));
    }
    TBregWrite(devid, 0x88, off[devid]*4); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000008); /* direction (0x8:PCI <-- local) */

    rcnt = read_cbcounter(devid, 0);
    dmastart = size*ratio+off[devid];
    while (rcnt < dmastart)
    {
	rcnt = read_cbcounter(devid, 0);
    }

    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */
    off[devid] = off[devid] + size;
}



static void
danger_read_dualportram(int devid, int size, double ratio)
{
    register int rcnt = 0;
    register int dmastart;
	
    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "danger_read_dualportram(): offset+size too large.\n");
	exit(2);
    }

    TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */
    TBregWrite(devid, 0x88, off[devid]*4); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000008); /* direction (0x8:PCI <-- local) */

    rcnt = read_cbcounter(devid, 0);
    dmastart = size*ratio+off[devid];
    while (rcnt < dmastart)
    {
	rcnt = read_cbcounter(devid, 0);
    }

    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */
    waitfor(dmawwait*size);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }

    off[devid] = off[devid] + size;
}

static void
read_dualportram(int devid, int size)
{
    register int rcnt = 0;
    int ntry = 0;
    int rcntfin, rcnt2;

    /* wait until all data are written to dual-port ram on PHIB */
    rcntfin = off[devid] + size;

    while (rcnt != rcntfin)
    {
	rcnt = read_cbcounter(devid, 2);
	if (rcnt == rcntfin)
	{
	    break;
	}

/*
	if (rcnt != rcnt0)
	{
	    fprintf(stderr, "rcnt: %d  rcntfin: %d\n", rcnt, rcntfin);
	    rcnt0 = rcnt;
	}
	*/
    }
    Cfprintf(stderr, "rcnt reached rcntfin\n");

    do
    {
	if (ntry)
	{
	    rcnt = read_cbcounter(devid, 2);
	}

#if NODMA /* PIO read transfer */
	read_dualportramPIO(devid, size);
#else /* DMA write transfer */
	read_dualportramDMA(devid, size);
#endif

	ntry++;
	rcnt2 = read_cbcounter(devid, 0);
    } while (rcnt != rcnt2 && ntry < NTRYMAX);

    if (rcnt != rcnt2)
    {
	fprintf(stderr, "read_dualportram(): cannot correct. abort.\n");
	exit (1);
    }
    off[devid] = off[devid] + size;
}

#define NTRYMAX2 (1000)
static int
read_dualportram_timeout(int devid, int size)
{
    int ret;
    register int rcnt = 0;
    int ntry = 0;
    int rcntfin, rcnt2;

    /* wait until all data are written to dual-port ram on PHIB */
    rcntfin = off[devid] + size;
    for (ntry = 0; ntry < NTRYMAX2; ntry++)
    {
	rcnt = read_cbcounter(devid, 2);
	if (rcnt == rcntfin)
	{
	    break;
	}

/*
	if (rcnt != rcnt0)
	{
	    fprintf(stderr, "rcnt: %d  rcntfin: %d\n", rcnt, rcntfin);
	    rcnt0 = rcnt;
	}
	*/
    }
    ret = rcnt-off[devid];
    if (ntry == NTRYMAX2)
    {
	fprintf(stdout, "read %d words out of %d words **\n",
		rcnt-off[devid], rcntfin-off[devid]);
    }

    do
    {
	if (ntry)
	{
	    rcnt = read_cbcounter(devid, 2);
	}
	read_dualportramDMA(devid, size);
	ntry++;
	rcnt2 = read_cbcounter(devid, 0);
    } while (rcnt != rcnt2 && ntry < NTRYMAX);

    if (rcnt != rcnt2)
    {
	fprintf(stderr, "read_dualportram(): cannot correct. abort.\n");
	exit (1);
    }
    off[devid] = off[devid] + size;
    return(ret);
}



static void
read_dualportramDMA(int devid, int size)
{
    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "read_dualportramDMA(): offset+size too large.\n");
	exit(2);
    }

    TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */
    TBregWrite(devid, 0x88, off[devid]*4); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000008); /* direction (0x8:PCI <-- local) */
    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */
    waitfor(dmawwait*size);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
}

static void
read_dualportramPIO(int devid, int size)
{
    int k;
    unsigned int * mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_piowMC: open phib%d first.\n", devid);
	exit (2);
    }

    if (off[devid]+size > RAMSIZE)
    {
	fprintf(stderr, "read_dualportramPIO(): offset+size too large.\n");
	exit(2);
    }

    for (k = 0; k < size; k++)
    {
	resbufp[devid][k] = mp[off[devid]+k];
    }
}

void
phib_dmartestMC(int devid, int size, unsigned int *buf)
{
    int i;
    unsigned int * mp;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_pioconfwMC(): open phib%d first.\n", devid);
	exit(2);
    }

    if (size > RAMSIZE)
    {
	fprintf(stderr, "phib_dmarMC(): size too large.\n");
	exit(2);
    }

    if ((unsigned int *)resbufp[devid] != buf)
    {
	for (i = 0; i < size; i++)
	{
	    ((unsigned int *)(resbufp[devid]))[i] = buf[i];
	}
    }

    TBregWrite(devid, 0x84, resbufpa[devid]); /* start address in PCI space */
    TBregWrite(devid, 0x88, 0); /* start address in local space */
    TBregWrite(devid, 0x8c, size*4); /* size in byte */
    TBregWrite(devid, 0x90, 0x00000000); /* direction (0:PCI --> local) */
    wait_dmar_assertion(devid);
    TBregWrite(devid, 0xa8, 0x00000003); /* start DMA */

    waitfor(dmarwait*size);
    while (!(TBregRead(devid, 0xa8) & 0x10))
    {
    }
    off[devid] += size;
    TBmemWriteInt(LBCADDR, off[devid]);
    wait_dmar_deassertion(devid);
}

void
phib_dmawtestMC(int devid, int size, unsigned int *buf)
{
    int i;

    if (size > RAMSIZE)
    {
	fprintf(stderr, "phib_dmawMC(): size too large.\n");
	exit (1);
    }

    off[devid] = 0;
    read_dualportramDMA(devid, size);

    /* copy contents of 'resbufp[devid]' to 'buf' buffer
       specified by the user of this library */
    if ((unsigned int *)resbufp[devid] != buf)
    {
	for (i = 0; i < size; i++)
	{
	    buf[i] = ((unsigned int *)resbufp[devid])[i];
	}
    }
}

/*************  functions below should not be modified  *************/

static void
clear_cbcounter(int devid, int checkmax)
{
    off[devid] = RAMADDR;
    TBmemRead(devid, LBCADDR|CLR);
}

static int
read_cbstatus(int devid, unsigned int query)
{
    int data;

    data = TBmemRead(devid, query);
    data = STSBIT & data;

    if (data > 0)
    {
	data = 1;
    }
    return (data);
}

static int
read_cbcounter(int devid, int checkmax)
{
    int check;
    int data, data1;

    data = TBmemRead(devid, LBCADDR);
    MB;
    data = 0x00000fff & data;
    for (check = 0; check < checkmax; check++)
    {
	data1 = TBmemRead(devid, LBCADDR);
	MB;
	data1 = 0x00000fff & data1;
	if (data != data1)
	{
	    /* counter read error */
	    Cprintf("read_cbcounter data1: 0x%08x\n", data1);
	    data = -1;
	}
    }
    return (data);
}

/* wait until CB issues DMA-read request */
static void
wait_dmar_assertion(int devid)
{		
    Cfprintf(stderr, "waiting for assertion...\n");
			
    while (read_cbstatus(devid, LBCADDR|QUERY_DMAR) > 0)
    {
	MB;
	Cprintf("#a\n");
    }
    Cfprintf(stderr, "asserted\n");
}

static void
wait_dmar_deassertion(int devid)
{		
    Cfprintf(stderr, "waiting for deassertion...\n");

    Cprintf("## before counter: %d\n", read_cbcounter(devid, 1));
    while (read_cbstatus(devid, LBCADDR | QUERY_DMAR) == 0)
    {
	Cprintf("+++ counter: %d\n", read_cbcounter(devid, 1));
	MB;
	Cprintf("#d\n");
    }
    Cprintf("## after counter: %d\n", read_cbcounter(devid, 1));
    Cfprintf(stderr, "deasserted\n");
}

#if NODMA
static void
testrun(int devid)
{
    int i;
    int *buf = NULL;
    unsigned int *mp = NULL;

    mp = phib[devid];
    if (!mp)
    {
	fprintf(stderr,	"phib_piowMC: open phib%d first.\n", devid);
	exit (2);
    }

    if (buf == NULL)
    {
	buf = (int *)calloc(sizeof(unsigned int), RAMSIZE);
    }

    for (i = 0; i < RAMSIZE; i++) {
	buf[i] = 0x12345678;
    }
    for (i = 0; i < RAMSIZE; i++) {
	TBmemWriteInt(i, i*4);
    }
    for (i = 0; i < RAMSIZE; i++) {
	buf[i]=mp[i];
    }

    for (i = 0; i < RAMSIZE; i++) {
	if (i*4 != buf[i]) {
	    printf("buf[0x%04x]: 0x%08x\n", i, buf[i]);
	}
    }
    if (buf) {
	free(buf);
	buf = NULL;
    }
}
#else /* DMA */
static void
testrun(int devid)
{
    int j;
    int err = 0;
    int *buf = NULL;

    if (buf == NULL)
    {
	buf = (int *)calloc(sizeof(unsigned int), RAMSIZE);
    }

    Cfprintf(stderr, "sharedbufp: %08x  resbufp[%d]: %08x\n",
	    sharedbufp, devid, resbufp[devid]);
    Cfprintf(stderr, "resbufpa[%d]: %08x\n", devid, resbufpa[devid]);
    Cfprintf(stderr, "RAMSIZE: %08x words\n", RAMSIZE);
    Cfprintf(stderr, "DMA size: %08x bytes\n", RAMSIZE*sizeof(unsigned int));

    while (1)
    {
	TBregWrite(devid, 0x80, 0x000001c3);
	TBregWrite(devid, 0x84, resbufpa[devid]+0x0); /* start address in PCI space */
	TBregWrite(devid, 0x88, 0x00000000);
	TBregWrite(devid, 0x8c, RAMSIZE*sizeof(unsigned int));

	for (j = 0; j < RAMSIZE; j++)
	{
	    buf[j] = j*4;
	    *((int *)resbufp[devid]+j) = 0xf;
	}
	memcpy(resbufp[devid], buf, RAMSIZE*sizeof(int));

    /* write */
	TBregWrite(devid, 0x90, 0x00000000);
	TBregWrite(devid, 0xa8, 0x00000003);

	Cfprintf(stderr, "testrun 0\n");


	Cprintf("0x18: %08x\n", TBregRead(devid, 0x18));

	while (!(TBregRead(devid, 0xa8) & 0x10))
	{
	}

	Cfprintf(stderr, "testrun 1\n");

	memset(resbufp[devid], 0, RAMSIZE*sizeof(int));

	/* read */
	TBregWrite(devid, 0x90, 0x00000008);
	TBregWrite(devid, 0xa8, 0x00000003);
	while (!(TBregRead(devid, 0xa8) & 0x10))
	{
	}

	for (j = 0; j < RAMSIZE; j++)
	{
	    if (buf[j] != resbufp[devid][j])
	    {
		fprintf(stderr, "buf[%0d]: 0x%0x  rbp[0x%0x]: 0x%0x\n",
			j, buf[j], j, resbufp[devid][j]);
		err = 1;
	    }
	}
	if (err)
	{
	    fprintf(stderr, "tried to load DMA map but failed.\n");
	    exit(1);
/*
	    sharedpa[devid] =
		TBdmaMapLoad(devid, (unsigned int *)sharedbufp, RAMSIZE);
	    resbufpa[devid] = sharedpa[devid];
	    */
	}
	else
	{
	    break;
	}
    }
    if (buf) {
	free(buf);
	buf = NULL;
    }
}

#endif

static void
waitfor(double maxcount)
{
    int i, j;

    for (j = 0; j < maxcount; j++)
    {
	for (i = 0; i < 10; i++)
	{
	}
    }
}

static int
mutexlock(void)
{
    int status = 0;
#ifdef THREAD_SAFE_PHIBLIB
    status = pthread_mutex_lock(&phibmutex); 
#endif /* THREAD_SAFE_PHIBLIB */
    return (status);
}

static int
mutexunlock(void)
{
    int status = 0;
#ifdef THREAD_SAFE_PHIBLIB
    status = pthread_mutex_unlock(&phibmutex); 
#endif /* THREAD_SAFE_PHIBLIB */
    return (status);
}

