#!@WISH@

# --
# Copyright (c) 1996 Picture Elements, Inc.
#    Stephen Williams
#
#    c/o Picture Elements, Inc.
#    777 Panoramic Way
#    Berkeley, CA, 94704
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#
# $Id: showpci.tcl,v 1.6 1997/05/20 23:57:09 steve Exp $
#
# $Log: showpci.tcl,v $
# Revision 1.6  1997/05/20 23:57:09  steve
#  Make the extension a shared object
#  Make filenames more 8.3 friendly
#
# Revision 1.5  1997/03/15 20:01:09  steve
#  secondary bus status.
#
# Revision 1.4  1997/03/15 19:48:55  steve
#  Change BAR display for bridges.
#
# Revision 1.3  1997/03/15 19:46:03  steve
#  Change BAR display for bridges.
#
# Revision 1.2  1997/02/22 00:35:45  steve
#  Use autoconf configuration
#
# Revision 1.1  1996/10/10 02:23:27  steve
#  Add a device configuration display.
#
#


# --
# This is a pretty little program for displaying the standard
# configuration space fields in an X display. The script takes no
# options.
#
# When it displays, it shows the configuration registers
# interpreted. The device id, vendor id and instance number are
# editable, and as you change the numbers the rest of the display
# reflects the entered value.

load @libpciconf.so@

# --
# Gain access to the PCI bus configuration space.
pciopen pci -r /dev/pciconf

# --
# These variables reflect the device that I'm looking at.
#
set device_id 0
set vendor_id 0
set instance  0
set primary_cmd "0x0000"
set primary_stat "0x0000"
set secondary_stat ""
set revision  0
set class_code "0x00 0x00 0x00"
set cacheline_size "off"
set bist "0x00"
set interrupt_pin "x"
set interrupt_line "x"
set minimum_grant "x"
set maximum_latency "x"

set bar_label(0) ""
set bar_label(1) ""
set bar_label(2) ""
set bar_label(3) ""
set bar_label(4) ""
set bar_label(5) ""

proc format_bar value {

    if {$value & 1} {

	set result [format "I/O Space: 0x%04x" [expr {$value & 0xfffffffc}]]

    } else {

	set base [expr {$value & 0xfffffff0}]
	set type [expr {$value & 0x06}]
	set pref [expr {$value & 0x08}]

	switch $type {
	    0 {set type "32bit memory"}
	    2 {set type "20bit memory"}
	    4 {set type "64bit memory"}
	    6 {set type "XXXXX memory"}
	}

	if {$base == 0} {
	    set result [format "%s disabled" "$type"]
	} else {
	    set result [format "%s at 0x%08x" "$type" $base]
	    if {$value & 0x08} {append result " (prefetchable)"}
	}
    }

    return $result
}

proc format_bridge_bus {value} {
    set pr_bus [expr {$value & 0xff}]
    set se_bus [expr {($value & 0xff00) / 256}]
    set su_bus [expr {($value & 0xff0000) / 0x10000}]
    set se_lat [expr {($value & 0xff000000) / 0x1000000}]

    set result [format "pri=%d sec=%d-%d lat=0x%02x" \
	    $pr_bus $se_bus $su_bus $se_lat]
    return $result
}

proc format_bridge_io {value} {
    set io_type [expr {$value & 0x0f}]
    set io_base [expr {$value & 0xf0}]
    set io_lim  [expr {$value & 0xff00}]
    set sec_status [expr {($value & 0xffff0000) / 0x10000}]

    return [format "%d bit I/O at 0x%04x - 0x%04x" \
	    [expr {16 * ($io_type + 1)}] [expr $io_base * 0x100] $io_lim]
}

proc format_bridge_mem {value} {
    set mem_base [expr {$value & 0xfffff}]
    set mem_lim  [expr {($value & 0xffff0000)}]

    return [format "Memory at 0x%08x - 0x%08x" \
	    [expr $mem_base * 0x10000] $mem_lim]
}

# --
# this function reads the PCI device and fills in all the variables
# to reflect the configuration space. It is most handy to call this
# whenever the ids are changed.

proc read_device {} {
    global device_id vendor_id instance
    global primary_cmd primary_stat secondary_stat \
	    revision class_code \
	    cacheline_size \
	    interrupt_pin interrupt_line \
	    minimum_grant maximum_latency bist bar bar_label

    catch {pci select $vendor_id $device_id $instance}

    set primary_cmd [format "0x%04x" [pci get_word 0x04]]

    set primary_stat [format "0x%04x" [pci get_word 0x06]]

    set revision [pci get_byte 0x08]

    set class_value [format "0x%02x 0x%02x 0x%02x" \
	    [pci get_byte 0xb] \
	    [pci get_byte 0xa] \
	    [pci get_byte 0x9]]
    switch -regexp $class_value {
	"0x00 0x00 0x00" {set class_str "legacy device"}
	"0x00 0x01 0x00" {set class_str "legacy VGA device"}

	"0x01 0x00 0x00" {set class_str "SCSI controller"}
	"0x01 0x01 0x.." {set class_str "IDE controller"}
	"0x01 0x80 0x00" {set class_str "mass storage controller"}
	"0x01 0x.. 0x.." {set class_str "mass storage controller?"}

	"0x02 0x.. 0x.." {set class_str "network controller?"}

	"0x03 0x.. 0x.." {set class_str "display controller?"}

	"0x04 0x.. 0x.." {set class_str "multimedia device?"}

	"0x05 0x00 0x00" {set class_str "RAM"}
	"0x05 0x01 0x00" {set class_str "Flash"}
	"0x05 0x80 0x00" {set class_str "Generic memory"}
	"0x05 0x.. 0x.." {set class_str "memory?"}

	"0x06 0x04 0x00" {set class_str "PCI-to-PCI bridge"}
	"0x06 0x.. 0x.." {set class_str "A BRIDGE?"}

	"0x07 0x.. 0x.." {set class_str "simple communication?"}

	"0x08 0x.. 0x.." {set class_str "system peripheral?"}

	"0x09 0x.. 0x.." {set class_str "input device?"}

	"0x0a 0x.. 0x.." {set class_str "docking station?"}

	"0x0b 0x.. 0x.." {set class_str "processor?"}

	"0x0c 0x.. 0x.." {set class_str "serial bus?"}

	default {set class_str "A what?"}
    }
    set class_code "$class_value $class_str"

    set value [pci get_byte 0x0c]
    if {$value == 0} {
	set cacheline_size "off"
    } else {
	set cacheline_size [format "%d bytes" [expr "$value * 32"]]
    }

    set interrupt_pin [pci get_byte 0x3d]
    switch $interrupt_pin {
	0 {set interrupt_pin "none"}
	1 {set interrupt_pin "INTA#"}
	2 {set interrupt_pin "INTB#"}
	3 {set interrupt_pin "INTC#"}
	4 {set interrupt_pin "INTD#"}
    }

    set interrupt_line [pci get_byte 0x3c]

    set value [pci get_byte 0x3e]
    if {$value == 0} {
	set minimum_grant "none"
    } else {
	set minimum_grant \
	    [format "%d clocks (%f nSec)" $value [expr "30.3 * $value"]]
    }

    set value [pci get_byte 0x3f]
    if {$value == 0} {
	set maximum_latency "none"
    } else {
	set maximum_latency \
	    [format "%d clocks (%f nSec)" $value [expr "30.3 * $value"]]
    }

    set bist_value [pci get_byte 0xf]
    set bist_string [format "0x%02x" $bist_value]
    if {$bist_value & 0x80} {append bist_string " (BIST capable)"}
    if {$bist_value & 0x40} {append bist_string " (BIST start)"}
    if {$bist_value & 0x0f} {append bist_string [expr "$bist_value & 0x0f"]}
    set bist "$bist_string"

    set bar(0) [format_bar [pci get_dword 0x10]]
    set bar(1) [format_bar [pci get_dword 0x14]]

    if {[string compare $class_value "0x06 0x04 0x00"] == 0} {
	set bar(2) [format_bridge_bus [pci get_dword 0x18]]
	set bar(3) [format_bridge_io [pci get_dword 0x1C]]
	set bar(4) [format_bridge_mem [pci get_dword 0x20]]
	set bar(5) [format_bridge_mem [pci get_dword 0x24]]
	set bar_label(2) "Bus Info:"
	set bar_label(3) "Bridge I/O:"
	set bar_label(4) "Bridge Memory:"
	set bar_label(5) "Prefetchable:"

	set secondary_stat [format "0x%04x" [pci get_word 0x1e]]

    } else {
	set bar(2) [format_bar [pci get_dword 0x18]]
	set bar(3) [format_bar [pci get_dword 0x1C]]
	set bar(4) [format_bar [pci get_dword 0x20]]
	set bar(5) [format_bar [pci get_dword 0x24]]
	set bar_label(2) "BAR 2:"
	set bar_label(3) "BAR 3:"
	set bar_label(4) "BAR 4:"
 	set bar_label(5) "BAR 5:"

	set secondary_stat ""
   }
}

proc trace_device {name element op} { read_device }
trace variable device_id w trace_device
trace variable vendor_id w trace_device
trace variable instance  w trace_device

read_device

frame .buttons
button .buttons.refresh -text "refresh" -command read_device
pack .buttons.refresh -side left

set label_width 15

frame .vendor_id
label .vendor_id.label -text "Vendor ID:" -width $label_width -anchor e
entry .vendor_id.value -textvariable vendor_id
pack  .vendor_id.label .vendor_id.value -side left -anchor w

frame .device_id
label .device_id.label -text "Device ID:" -width $label_width -anchor e
entry .device_id.value -textvariable device_id
pack  .device_id.label .device_id.value -side left -anchor w

frame .instance
label .instance.label -text "Instance:" -width $label_width -anchor e
entry .instance.value -textvariable instance
pack  .instance.label .instance.value -side left -anchor w

frame .primary_cmd
label .primary_cmd.label -text "Command:" -width $label_width -anchor e
label .primary_cmd.value -textvariable primary_cmd
pack  .primary_cmd.label .primary_cmd.value -side left -anchor w

frame .primary_stat
label .primary_stat.label -text "Status:" -width $label_width -anchor e
label .primary_stat.value -textvariable primary_stat
pack  .primary_stat.label .primary_stat.value -side left -anchor w

frame .secondary_stat
label .secondary_stat.label -text "Secondary status:" -width $label_width -anchor e
label .secondary_stat.value -textvariable secondary_stat
pack  .secondary_stat.label .secondary_stat.value -side left -anchor w

frame .revision
label .revision.label -text "Revision ID:" -width $label_width -anchor e
label .revision.value -textvariable revision
pack  .revision.label .revision.value -side left -anchor w

frame .class_code
label .class_code.label -text "Class code:" -width $label_width -anchor e
label .class_code.value -textvariable class_code
pack  .class_code.label .class_code.value -side left

frame .cacheline_size
label .cacheline_size.label -text "Cachline Size:" -width $label_width -anchor e
label .cacheline_size.value -textvariable cacheline_size
pack  .cacheline_size.label .cacheline_size.value -side left

frame .bist
label .bist.label -text "BIST:" -width $label_width -anchor e
label .bist.value -textvariable bist
pack  .bist.label .bist.value -side left

frame .interrupt
label .interrupt.label_pin -text "Interrupt pin:" -width $label_width -anchor e
label .interrupt.value_pin -textvariable interrupt_pin
label .interrupt.label_line -text "line:"
label .interrupt.value_line -textvariable interrupt_line
pack  .interrupt.label_pin .interrupt.value_pin .interrupt.label_line .interrupt.value_line -side left

frame .grant
label .grant.label -text "Minimum Grant:" -width $label_width -anchor e
label .grant.value -textvariable minimum_grant
pack  .grant.label .grant.value -side left

frame .latency
label .latency.label -text "Max Latency:" -width $label_width -anchor e
label .latency.value -textvariable maximum_latency
pack  .latency.label .latency.value -side left

frame .bar0
label .bar0.label -text "BAR 0:" -width $label_width -anchor e
label .bar0.value -textvariable bar(0)
pack  .bar0.label .bar0.value -side left

frame .bar1
label .bar1.label -text "BAR 1:" -width $label_width -anchor e
label .bar1.value -textvariable bar(1)
pack  .bar1.label .bar1.value -side left

frame .bar2
label .bar2.label -textvariable bar_label(2) -width $label_width -anchor e
label .bar2.value -textvariable bar(2)
pack  .bar2.label .bar2.value -side left

frame .bar3
label .bar3.label -textvariable bar_label(3) -width $label_width -anchor e
label .bar3.value -textvariable bar(3)
pack  .bar3.label .bar3.value -side left

frame .bar4
label .bar4.label -textvariable bar_label(4) -width $label_width -anchor e
label .bar4.value -textvariable bar(4)
pack  .bar4.label .bar4.value -side left

frame .bar5
label .bar5.label -textvariable bar_label(5) -width $label_width -anchor e
label .bar5.value -textvariable bar(5)
pack  .bar5.label .bar5.value -side left

# --
# Fine, glue everything together in one pretty new display
#
pack .buttons \
	.vendor_id \
	.device_id \
	.instance \
	.primary_cmd \
	.primary_stat \
	.secondary_stat \
	.revision \
	.class_code \
	.cacheline_size \
	.bist \
	.interrupt \
	.grant \
	.latency \
	.bar0 .bar1 .bar2 .bar3 .bar4 .bar5\
	-anchor w
