/*
 * Copyright (c) 1996 Picture Elements, Inc.
 *    Stephen Williams
 *
 *    c/o Picture Elements, Inc.
 *    777 Panoramic Way
 *    Berkeley, CA, 94704
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */
#ident "$Id: pciconf.c,v 1.4 1997/02/22 00:35:45 steve Exp $"
/*
 * $Log: pciconf.c,v $
 * Revision 1.4  1997/02/22 00:35:45  steve
 *  Use autoconf configuration
 *
 * Revision 1.3  1996/10/01 00:47:13  steve
 *  Copy notices.
 *
 * Revision 1.2  1996/09/26 21:15:10  steve
 *  Count references to module.
 *
 * Revision 1.1.1.1  1996/09/23 20:20:22  steve
 *  Initial release.
 *
 */

# include  <linux/module.h>
# include  <linux/bios32.h>
# include  <linux/fs.h>
# include  <linux/mm.h>
# include  <linux/miscdevice.h>
# include  <asm/segment.h>

# include  "pciconf.h"


# define SPACE_LIMIT 256


# define PCICONF "pciconf"

/*
 * This device has no minor numbers for options, so I choose to make
 * this a ``misc'' device. This means it has a major number of the
 * misc device, like lost of other device drivers, and the minor
 * number is used to seperate them. The pciconf_minor contains the
 * misc minor number I choose to use.
 */

static int pciconf_minor = MISC_MINOR;

struct Instance {

	/* These values represent a device that the process is
	   interested in. The device may or may not exist. */

      struct pciconf_info info;

	/* These are details about the device I found that matches the
	   above parameters. Use these to actually manipulate the
	   device of interest. these values are useable if the
	   valid_flag it true. */

      unsigned char bus, device_fn;

      int valid_flag;
};


/*
 * Use the device information to locate the board/function and set the
 * valid_flag if the information exists.
 */
static void match_device(struct Instance*xsp)
{
      int rc = pcibios_find_device(xsp->info.vendor_id,
				   xsp->info.device_id,
				   xsp->info.instance,  &xsp->bus,
				   &xsp->device_fn);

      if (rc == PCIBIOS_SUCCESSFUL) xsp->valid_flag = 1;
      else xsp->valid_flag = 0;
}

/*
 * Open allocates an instance structure and attaches it to the file
 * structure for the process opening me. This gives the process the
 * impression that every open leads to a unique device.
 *
 * xxrelease() releases the allocated structure.
 */

static int xxopen(struct inode *inode, struct file *file)
{
      struct Instance*xsp;

      xsp = (struct Instance*)kmalloc(sizeof(*xsp), GFP_KERNEL);
      if (xsp == 0) return -ENOMEM;

      xsp->info.vendor_id = 0;
      xsp->info.device_id = 0;
      xsp->info.instance = 0;

      xsp->valid_flag = 0;

      file->private_data = xsp;
      file->f_pos = 0;

      MOD_INC_USE_COUNT;

      return 0;
}

static void xxrelease(struct inode *inode, struct file *file)
{
      struct Instance*xsp = (struct Instance*)file->private_data;

      kfree(xsp);

      MOD_DEC_USE_COUNT;
}

static int xxlseek (struct inode *i, struct file *file, off_t off, int wh)
{
      struct Instance*xsp = (struct Instance*)file->private_data;

      off_t dest;

      switch (wh) {
	  case 0: /* SEEK_SET */
	    dest = off;
	    break;
	  case 1: /* SEEK_CUR */
	    dest = file->f_pos + off;
	    break;
	  case 2: /* SEEK_END */
	    dest = SPACE_LIMIT + off;
	    break;

	  default:
	    return -EINVAL;
      }

      if (dest < 0) dest == 0;
      if (dest > SPACE_LIMIT) dest = SPACE_LIMIT;
      file->f_pos = dest;

      return file->f_pos;
}


static int xxread(struct inode *inode, struct file *file,
		  char *bytes, int count)
{
      struct Instance*xsp = (struct Instance*)file->private_data;

      int tcount = count;

      if (! xsp->valid_flag) return -EIO;

      while ((tcount > 0) && (file->f_pos < SPACE_LIMIT)) {
	    int rc;
	    unsigned char val;

	    rc = pcibios_read_config_byte(xsp->bus, xsp->device_fn,
					  file->f_pos, &val);

	    put_user(val, bytes);
	    bytes += 1;
	    file->f_pos += 1;
	    tcount -= 1;
      }

      return count - tcount;
}


static int xxwrite(struct inode *inode, struct file *file,
		   const char*bytes, const int count)
{
      struct Instance*xsp = (struct Instance*)file->private_data;

      int tcount = count;
      if ((file->f_pos + tcount) > SPACE_LIMIT)
	    tcount = SPACE_LIMIT - file->f_pos;

      if (! xsp->valid_flag) return -EIO;

      while ((tcount > 0) && (file->f_pos < SPACE_LIMIT)) {
	    int rc;
	    int wrote;
	    const long f_pos = file->f_pos;

	    if ((tcount >= 4) && (f_pos%4 == 0)) {
		  unsigned long val = get_user((unsigned long*)bytes);

		  rc = pcibios_write_config_dword(xsp->bus,
						  xsp->device_fn,
						  f_pos, val);
		  wrote = 4;

	    } else if ((tcount >= 2) && (f_pos%2 == 0)) {
		  unsigned short val = get_user((unsigned short*)bytes);

		  rc = pcibios_write_config_word(xsp->bus,
						 xsp->device_fn,
						 f_pos, val);
		  wrote = 2;

	    } else {
		  unsigned char val = get_user(bytes);

		  rc = pcibios_write_config_byte(xsp->bus,
						 xsp->device_fn,
						 f_pos, val);
		  wrote = 1;
	    }

	    bytes += wrote;
	    file->f_pos += wrote;
	    tcount -= wrote;
      }

      return count - tcount;
}


static int xxioctl(struct inode *inode, struct file *file,
		   unsigned int cmd, unsigned long arg)
{
      struct Instance*xsp = (struct Instance*)file->private_data;

      switch (cmd) {
	  case PCICONF_SELECT:
	    memcpy_fromfs(&xsp->info, (void*)arg, sizeof xsp->info);
	    match_device(xsp);
	    file->f_pos = 0;

	    if (xsp->valid_flag) return 0;
	    else return -ENODEV;
      }

      return -ENOTTY;
}


static const struct file_operations ops = {
      xxlseek,
      xxread,
      xxwrite,
      0,
      0,
      xxioctl,
      0,
      xxopen,
      xxrelease,
      0,
      0,
      0,
      0
};

static struct miscdevice misc = {
      0,
      PCICONF,
      &ops
};

int init_module(void)
{
      int rc;

      misc.minor = pciconf_minor;
      rc = misc_register(&misc);

      return 0;
}

void cleanup_module(void)
{
      misc_deregister(&misc);
}
