/*
 * PREDICTOR.C
 *
 * block float  predictor pipeline
 *
 * Copyright Jun Makino 1997
 *
 * Version 1.0 Nov 5 1997
 * Version 1.1 Mar 2 1998
 *
 * Major changes:
 *    -- test mode added
 *
 * Version 2.0 Apr 29 1998
 * -- Major changes: a2dot now reduced to 10+1 bits! 
 *
 *
 * Developping notes
 *
 * * convert_predictor **
 * -- x to ix seems okay (just shifting... )
 */
#include "grape6sim.h"

static int mkmintab[5] = {0,1,6,10,16};
/*last term changed to 16 from 14 on Apr 29, 1998 */
static int lentab[5] = {LONGBITS, PRED_V_LEN, PRED_A_LEN,
			    PRED_A1_LEN, PRED_A2_LEN};
    

/* convert_predictor_time
   determine predictor exponent  time and timestep
   
   *THIS FUNCTION IS NOT PART OF GRAPE-6 CHIP */


LONG convert_predictor_time(ULONG *  tjlsb, /* LSB of tj */
			    ULONG * dtjmsb, /* location of MSB of delta ti */
			    LONG * dtexp, /* exponent of dt in physical unit */
			    double rtj, /* particle time */
			    double rdtj, /* particle timestep */
			    ULONG tunit /* time resolution=2**-tinit*/)
{
  
  /* first, we calculate time values */
  double dtmin, xmin, dtfrac;
  int idtexp, dtminexp;
  int e0btmp, ek0, e0b, expv;
  ULONG t_int;
  int i,k;
  
  dtmin = 1.0/(((ULONG)1)<<tunit);
  dtfrac = frexp(rdtj, &idtexp);
  *dtexp = 1-idtexp;
  if(dtfrac != 0.5){
    fprintf(stderr, "convert_predictor: dtfrac  = %le != 0.5!\n");
    return -1;
  }
  frexp(dtmin, &dtminexp);
  *dtjmsb = idtexp - dtminexp;
  t_int = rtj/rdtj;
  if (t_int * rdtj != rtj){
    fprintf(stderr, "tj not multiple of dtj %le %le %le\n",
	    rtj, rdtj, rtj/rdtj);
    return -1;
  }
  *tjlsb = (ULONG) t_int & (ULONG) 1;
  dprintf(4,"tunit, dt, dtjmsb, tjlsb = %ld %le %ld %lx\n",
	  tunit, rdtj, *dtjmsb, *tjlsb);
  return 0;
}

ULONG determine_predictor_exponent(LONG * e0b,
				   LONG dtexp,
				   double x[5], /* position, vel ... */
				   ULONG xunit /* position resolution= 2**-xunit*/)
{
  
  /* since we use common exponent for a2-v, they should be
     calculated in the same way, except for the difference in the
     power of dt.
     
     For velocity, we have the exponent same as the original velocity,
     since velocity is already normalized and dt does not exceed one.
     In order to prevent possible overflow, mantissa of v should be
     downshifted (to use only 23 bits).
     */
  
  double xmin, dtfrac;
  int e0btmp, ek0,  expv;
  ULONG t_int;
  int i,k;
  
  dprintf(4,"dtexp = %ld\n", dtexp);
  *e0b = -1000000000;
  for(i=1;i<5;i++){
    if (x[i] != 0.0){
      frexp(x[i], &ek0);
      e0btmp = ek0 - dtexp*i + mkmintab[i];
      if ((*e0b < e0btmp)) *e0b = e0btmp;
    }
    dprintf(4,"i, ek0, e0btmp, e0b  = %ld %ld %ld %ld\n", i,ek0,e0btmp,*e0b);
  }
  return 0;
}

ULONG convert_predictor_using_e0b(ULONG ix[5], /* converted position, vel... */
				  double x[5], /* position, vel ... */
				  LONG e0b,
				  LONG dtexp,
				  ULONG xunit /* position resolution= 2**-xunit*/)

{
  double xmin, dtfrac;
  int e0btmp, ek0,  expv;
  ULONG t_int;
  int i,k;

  dprintf(4, "using_e0b: e0b, dtexp = %ld %ld\n", e0b, dtexp);
  /* convert x and set exponent of v */
  ix[0] = CONVERT_DOUBLE_TO_GRAPE_INT_POS(x[0],xunit);
  expv = e0b+xunit;
  dprintf(4,"expv = %ld\n", expv);
  if(expv > 0){
    for(i=1;i<5;i++){
      int msbloc, sign;
      double scaled_pk;
      scaled_pk = frexp(x[i], &ek0);
      if (scaled_pk < 0.0){
	sign = 1;
	scaled_pk = - scaled_pk;
      }else{
	sign = 0;
      }
      
      msbloc = PRED_V_LEN +ek0-e0b-dtexp*i;
      ix[i] = (ULONG)rint(ldexp(scaled_pk, msbloc));
      ix[i] |= sign <<lentab[i];
      dprintf(4,"i, scaled_pk, msbloc, ix  = %ld %le %ld %lx\n",
	      i,scaled_pk, msbloc, ix[i]);
    }
    
    
    /*    expv += PRED_V_EXP_OFFSET;*/
    ix[1] |= ((ULONG)expv) << (lentab[1]+1);
  }else{
        for(i=1;i<5;i++)ix[i] = 0;
  }
  return 0;
}



ULONG convert_predictor(ULONG *  tjlsb, /* LSB of tj */
			ULONG * dtjmsb, /* location of MSB of delta ti */
			ULONG ix[5], /* converted position, vel... */
			double rtj, /* particle time */
			double rdtj, /* particle timestep */
			double x[5], /* position, vel ... */
			ULONG xunit, /* position resolution= 2**-xunit*/
			ULONG tunit /* time resolution=2**-tinit*/)
{
  
  /* since we use common exponent for a2-v, they should be
     calculated in the same way, except for the difference in the
     power of dt.
     
     For velocity, we have the exponent same as the original velocity,
     since velocity is already normalized and dt does not exceed one.
     In order to prevent possible overflow, mantissa of v should be
     downshifted (to use only 23 bits).
     */
  
  /* first, we calculate time values */
  double dtmin, xmin, dtfrac;
  LONG dtexp,e0b;
  int e0btmp, ek0,  expv;
  ULONG t_int;
  int i,k;
  ULONG err;
  err = convert_predictor_time(tjlsb,dtjmsb, &dtexp,  rtj, rdtj,tunit);
  err |= determine_predictor_exponent(&e0b, dtexp,  x, xunit);
  err |= convert_predictor_using_e0b(ix,  x,  e0b,  dtexp,  xunit);
  return err;
}


  
ULONG convert_predictor_vector(ULONG *  tjlsb, /* LSB of tj */
			ULONG * dtjmsb, /* location of MSB of delta ti */
			ULONG ix[3][5], /* converted position, vel... */
			double rtj, /* particle time */
			double rdtj, /* particle timestep */
			double x[3], /* position, vel ... */
			double v[3], /* position, vel ... */
			double aby2[3], /* position, vel ... */
			double a1by6[3], /* position, vel ... */
			double a2by18[3], /* position, vel ... */
			ULONG xunit, /* position resolution= 2**-xunit*/
			ULONG tunit /* time resolution=2**-tinit*/)
{
  
  /* since we use common exponent for a2-v, they should be
     calculated in the same way, except for the difference in the
     power of dt.
     
     For velocity, we have the exponent same as the original velocity,
     since velocity is already normalized and dt does not exceed one.
     In order to prevent possible overflow, mantissa of v should be
     downshifted (to use only 23 bits).
     */
  
  /* first, we calculate time values */
  double dtmin, xmin, dtfrac;
  LONG dtexp,e0b;
  double xdata[3][5];
  LONG e0btmp;
  int i,k;
  ULONG err;
  err = convert_predictor_time(tjlsb,dtjmsb, &dtexp,  rtj, rdtj,tunit);
  e0b = -10000000000;
  for(k=0;k<3;k++){
    
    xdata[k][0] = x[k];
    xdata[k][1] = v[k];
    xdata[k][2] = aby2[k];
    xdata[k][3] = a1by6[k];
    xdata[k][4] = a2by18[k];
    err |= determine_predictor_exponent(&e0btmp, dtexp,  xdata[k], xunit);
    if (e0btmp > e0b)e0b = e0btmp;
  }
  for(k=0;k<3;k++){
    err |= convert_predictor_using_e0b(ix[k],  xdata[k],  e0b,  dtexp,  xunit);
  }
  dprintf(2,"(convert_predictor) xunit, tunit, tj, dtj : %ld  %ld  %ld  %ld\n",
	  xunit, tunit, *tjlsb, *dtjmsb);
  for(k=0;k<5;k++){
    dprintf(2,"(convert .. )k, ix = %ld %lx %lx %lx\n", k, ix[k][0],ix[k][1],ix[k][2]);
  }
  return err;
}


  

/* convert_predicted_result
   convert predictor output in GRAPE internal format
   back to usual DOUBLE PREC format.
   THIS FUNCTION IS NOT PART OF GRAPE-6 CHIP */

ULONG convert_predicted_result(double * xp,
			       double * vp,
			       LONG ixp,
			       ULONG ivp,
			       ULONG xunit, /* position resolution= 2**-xunit*/
			       ULONG tunit /* time resolution  */)
{
    double tmp, sgn ;
    int binary_point, vexp, sign;
    tmp = ixp;
    binary_point = -xunit;
    /*    *xp= ldexp(tmp, binary_point);*/
    *xp= CONVERT_GRAPE_INT_POS_TO_DOUBLE(ixp,xunit);
    tmp = convert_grape_float_to_double(ivp, INTERACTION_F_LEN_U);
    dprintf(4,"tmp = %le\n", tmp, *vp);
    *vp = ldexp(tmp, -((int)(xunit + 0*PRED_V_EXP_OFFSET-tunit)));
    dprintf(4,"tmp, vp, xunit = %le %le %ld\n", tmp, *vp, xunit);
    return 0;
}

	
/* predict
   predictor pipeline simulator
   mode=0: pipeline for position
        1: for velocity
	*/
		 
    
ULONG predict_without_test(ULONG * result, /* result  */
	      ULONG mode, /* 0: position, 1: velocity */
	      ULONG ti, /* present time */
	      ULONG tjlsb, /* LSB of ti */
	      ULONG dtjmsb, /* location of MSB of delta ti */
	      ULONG a2by18, /* a2dot divided by 18 */
	      ULONG a1by6, /* a1dot divided by 6 */
	      ULONG aby2, /* a divided by 2 */
	      ULONG v, /* velocity */
	      ULONG x /* position */)
     /*
      * data format for a2,a1,a : sign + magnitude, sign: 1=minus
      * data format for v : exponent + sign + magnitude
      */
{
    ULONG dt, err, val, vexp, vsign ;
    err = tcalc(&dt, ti, tjlsb, dtjmsb); 
    dprintf(4, "(predict) dt = %lx\n", dt);
    err |=   pmult_with_sign(&val, dt, a2by18, PRED_A2_LEN, PRED_A2_LEN);
    dprintf(4, "(predict) a2term = %lx\n", val);
    
    if (mode == 0){
	err |= pmult_with_sign_by_constant(&val, val, MUL_3BY4,
					   PRED_A2_LEN, PRED_A2_LEN);
    }
    err |= padd_with_sign(&val, val, PRED_A2_LEN, a1by6,
			  PRED_A1_LEN,  PRED_A1_LEN);
    dprintf(4, "(predict) after add a1 = %lx\n", val);
    if (mode == 1){
	err |= pmult_with_sign_by_constant(&val, val, MUL_3BY2,
					   PRED_A1_LEN,  PRED_A1_LEN);
    dprintf(4, "(predict) after 1.5x = %lx\n", val);
    }
    err |=   pmult_with_sign(&val, dt, val, PRED_A1_LEN, PRED_A1_LEN);
    err |= padd_with_sign(&val, val, PRED_A1_LEN, aby2,
			  PRED_A_LEN,  PRED_A_LEN);
    dprintf(4, "(predict) after add a = %lx\n", val);
    if (mode == 1){
	err |= pmult_with_sign_by_constant(&val, val, MUL_2,
					   PRED_A_LEN,  PRED_A_LEN);
    dprintf(4, "(predict) after 2x = %lx\n", val);
    }
    err |=   pmult_with_sign(&val, dt, val, PRED_A_LEN, PRED_A_LEN);
    err |= padd_with_sign(&val, val, PRED_A_LEN, v,
			  PRED_V_LEN,  PRED_V_LEN);
    dprintf(4, "(predict) after add v = %lx\n", val);
    vexp = (v>>(PRED_V_LEN+1))&((1<<PRED_V_EXP_LEN)-1);
    dprintf(4, "(predict) vexp = %lx\n", vexp);
    
    if (mode == 0){
      /* position prediction */
	ULONG vmag;
	err |=   pmult_with_sign(&val, dt, val, PRED_V_LEN, PRED_V_LEN);
	dprintf(4, "(predict) after mul dt = %lx\n", val);
	vsign = (val>>PRED_V_LEN)&1;
	vmag = (val&((1<<PRED_V_LEN)-1))<<(LONGBITS-PRED_V_LEN);
	dprintf(4, "(predict) vsign, vmag  = %lx %lx\n", vsign, vmag);
	if (vexp > 0){
	    vmag = force_1_round_and_shift(vmag, LONGBITS,
					   vexp);
	}else{
	    vmag = (ULONG) 0;
	}
	dprintf(4, "(predict)  vmag  after shift, x = %lx %lx\n",  vmag, x);
	if (vsign){
	    *result = x - vmag;
	}else{
	    *result = x + vmag;
	}
    }else{
      /* velocity prediction */
      /* first, normalize result val */
      int iexp, loc;
      ULONG sign, exponent, mantissa, zero;
      sign = (val>>PRED_V_LEN) & ULONG_ONE;
      zero = 0;
      mantissa = (val & ((1<<(PRED_V_LEN))-1));
      loc = (int) priority_encoder(mantissa)+1;
      iexp = loc - PRED_V_LEN -dtjmsb ;

      dprintf(4, "vloc, iexp, vexp, dtjmsb = %ld %ld %ld %ld\n", loc, iexp, vexp,(int) dtjmsb);
      if(mantissa == ULONG_ZERO){
	zero = 1;
      }
      vexp += iexp + INTERACTION_POSITION_EXP_OFFSET;
      mantissa = force_1_round_and_shift(mantissa, loc, INTERACTION_F_LEN_U);
      *result =  compose_float( INTERACTION_F_LEN_U, vexp, sign, zero, mantissa);
    }
    dprintf(4, "result = %lx\n", *result);
    return err;
}

ULONG predict_with_testmode(ULONG * result, /* result  */
			    ULONG mode, /* 0: position, 1: velocity */
			    ULONG ti, /* present time */
			    ULONG tjlsb, /* LSB of ti */
			    ULONG dtjmsb, /* location of MSB of delta ti */
			    ULONG a2by18, /* a2dot divided by 18 */
			    ULONG a1by6, /* a1dot divided by 6 */
			    ULONG aby2, /* a divided by 2 */
			    ULONG v, /* velocity */
			    ULONG x,  /* position */
			    ULONG passk4, /* copy  A2BY18 to K4 */
			    ULONG passk4a, /* copy K4 to K4A */
			    ULONG passk3a, /* copy K3 to K3A */
			    ULONG passk2a, /* copy K2 to K2A */
			    ULONG passk1a, /* copy K1 to K1A */
			    ULONG passk3d, /* copy K3C to K3D */
			    ULONG passk3e, /* copy K3D to K3E */
			    ULONG passk2c, /* copy K2B to K2C */
			    ULONG passk2d) /* K2C to K2D */

    /*
     * data format for a2,a1,a : sign + magnitude, sign: 1=minus
     * data format for v : exponent + sign + magnitude
     */
{
    ULONG dt, err, val, vexp, vsign ;
    err = tcalc(&dt, ti, tjlsb, dtjmsb); 
    dprintf(4, "(predict) dt = %lx\n", dt);
    if (passk4 == 0){
      err |=   pmult_with_sign(&val, dt, a2by18, PRED_A2_LEN, PRED_A2_LEN);
    }else{
      val = a2by18;
    }

    dprintf(4, "(predict) a2term = %lx\n", val);
    
    if (mode == 0){
      if (passk4a == 0){
	err |= pmult_with_sign_by_constant(&val, val, MUL_3BY4,
					   PRED_A2_LEN, PRED_A2_LEN);
      }
    }
    err |= padd_with_sign(&val, val, PRED_A2_LEN, a1by6,
			  PRED_A1_LEN,  PRED_A1_LEN);
    dprintf(4, "(predict) after add a1 = %lx\n", val);
    if (mode == 1){
      if (passk3d == 0){
	err |= pmult_with_sign_by_constant(&val, val, MUL_3BY2,
					   PRED_A1_LEN,  PRED_A1_LEN);
      }
      dprintf(4, "(predict) after 1.5x = %lx\n", val);
    }
    if (((mode == 0) && (passk3a == 0))||((mode == 1) && (passk3e == 0) )){
      err |=   pmult_with_sign(&val, dt, val, PRED_A1_LEN, PRED_A1_LEN);
    }
    
    err |= padd_with_sign(&val, val, PRED_A1_LEN, aby2,
			  PRED_A_LEN,  PRED_A_LEN);
    dprintf(4, "(predict) after add a = %lx\n", val);
    if (mode == 1){
      if(passk2c == 0){
	err |= pmult_with_sign_by_constant(&val, val, MUL_2,
					   PRED_A_LEN,  PRED_A_LEN);
      }
      dprintf(4, "(predict) after 2x = %lx\n", val);
    }
    if (((mode == 0) &&(passk2a == 0))||((mode == 1) &&(passk2d == 0))){
      err |=   pmult_with_sign(&val, dt, val, PRED_A_LEN, PRED_A_LEN);
    }
    err |= padd_with_sign(&val, val, PRED_A_LEN, v,
			  PRED_V_LEN,  PRED_V_LEN);
    dprintf(4, "(predict) after add v = %lx\n", val);
    vexp = (v>>(PRED_V_LEN+1))&((1<<PRED_V_EXP_LEN)-1);
    dprintf(4, "(predict) vexp = %lx\n", vexp);
    
    if (mode == 0){
      /* position prediction */
	ULONG vmag;
	if (passk1a  == ULONG_ZERO){
	  err |=   pmult_with_sign(&val, dt, val, PRED_V_LEN, PRED_V_LEN);
	}
	dprintf(4, "(predict) after mul dt = %lx\n", val);
	vsign = (val>>PRED_V_LEN)&1;
	vmag = (val&((1<<PRED_V_LEN)-1))<<(LONGBITS-PRED_V_LEN);
	dprintf(4, "(predict) vsign, vmag  = %lx %lx\n", vsign, vmag);
	if (vexp > 0){
	    vmag = force_1_round_and_shift(vmag, LONGBITS,
					   vexp);
	}else{
	    vmag = (ULONG) 0;
	}
	dprintf(4, "(predict)  vmag  after shift, x = %lx %lx\n",  vmag, x);
	if (vsign){
	    *result = x - vmag;
	}else{
	    *result = x + vmag;
	}
    }else{
      /* velocity prediction */
      /* first, normalize result val */
      int iexp, loc;
      ULONG sign, exponent, mantissa, zero;
      sign = (val>>PRED_V_LEN) & ULONG_ONE;
      zero = 0;
      mantissa = (val & ((1<<(PRED_V_LEN))-1));
      loc = (int) priority_encoder(mantissa)+1;
      iexp = loc - PRED_V_LEN -dtjmsb ;

      dprintf(4, "vloc, iexp, vexp, dtjmsb = %ld %ld %ld %ld\n", loc, iexp, vexp, (int)dtjmsb);
      if(mantissa == ULONG_ZERO){
	zero = 1;
      }
      vexp += iexp + INTERACTION_POSITION_EXP_OFFSET;
      mantissa = force_1_round_and_shift(mantissa, loc, INTERACTION_F_LEN_U);
      *result =  compose_float( INTERACTION_F_LEN_U, vexp, sign, zero, mantissa);
    }
#ifdef X86
    dprintf(4, "result = %Lx\n", *result);
#else    
    dprintf(4, "result = %lx\n", *result); 
#endif
    return err;
}


ULONG predict(ULONG * result, /* result  */
	      ULONG mode, /* 0: position, 1: velocity */
	      ULONG ti, /* present time */
	      ULONG tjlsb, /* LSB of ti */
	      ULONG dtjmsb, /* location of MSB of delta ti */
	      ULONG a2by18, /* a2dot divided by 18 */
	      ULONG a1by6, /* a1dot divided by 6 */
	      ULONG aby2, /* a divided by 2 */
	      ULONG v, /* velocity */
	      ULONG x /* position */)
{
  return predict_with_testmode(result,  mode, ti, tjlsb, dtjmsb,
			       a2by18, a1by6, aby2, v, x,
			       ULONG_ZERO, ULONG_ZERO, ULONG_ZERO,
			       ULONG_ZERO, ULONG_ZERO, ULONG_ZERO,
			       ULONG_ZERO, ULONG_ZERO, ULONG_ZERO);
}

/* predict_double
   predictor function using DOUBLE PREC format
   NOT PART OF G6 CHIP */
ULONG predict_double(double * result, /* result  */
		     ULONG mode, /* 0: position, 1: velocity */
		     double ti, /* present time */
		     double tj, /* particle time */
		     double a2by18, /* a2dot divided by 18 */
		     double a1by6, /* a1dot divided by 6 */
		     double aby2, /* a divided by 2 */
		     double v, /* velocity */
		     double x /* position */)
{
  double pred, dt;
  dt = ti - tj;
  dprintf(4,"(predict_double) dt,x,v,a,a1,a2 = %le %le %le %le %le %le\n",
	 dt,x,v,aby2,a1by6,a2by18);
  if(mode == 0){
    pred = (((a2by18*dt*0.75+a1by6)*dt + aby2)*dt +v)*dt+ x;
  }else{
    pred =  ((a2by18*dt+a1by6)*dt*1.5+aby2)*2*dt+v;
  }
  *result = pred;
  return 0;
}

ULONG predict_vector_on_emulator(double xp[3], /* result  */
				 double vp[3], /* result  */
				 double ti, /* present time */
				 double tj, /* particle time */
				 double dtj, /* particle time */
				 int tunit,
				 int xunit,
				 double a2by18[3], /* a2dot divided by 18 */
				 double a1by6[3], /* a1dot divided by 6 */
				 double aby2[3], /* a divided by 2 */
				 double v[3], /* velocity */
				 double x[3] /* position */)
{
  ULONG tjlsb, dtjmsb, ix[3][5];
  ULONG utunit, uxunit, iti;
  ULONG ixp[3], ivp[3];
  int k;
  utunit = tunit;
  uxunit = xunit;
  convert_predictor_vector(&tjlsb, &dtjmsb, ix, tj, dtj,
			   x, v, aby2, a1by6, a2by18,
			   utunit, uxunit);
  iti = (ULONG) ldexp(ti, (int)tunit);
  for(k=0;k<3;k++){
    predict(ixp+k,ULONG_ZERO,iti,tjlsb, dtjmsb, ix[k][4],ix[k][3],ix[k][2],ix[k][1],ix[k][0]);
    predict(ivp+k,ULONG_ONE,iti,tjlsb, dtjmsb, ix[k][4],ix[k][3],ix[k][2],ix[k][1],ix[k][0]);
    convert_predicted_result(xp+k, vp+k,ixp[k], ivp[k],  xunit, tunit);

  }
  return 0;

}

ULONG predict_vector_on_emulator_(double xp[3], /* result  */
				 double vp[3], /* result  */
				 double *ti, /* present time */
				 double *tj, /* particle time */
				 double *dtj, /* particle time */
				 int *tunit,
				 int *xunit,
				 double a2by18[3], /* a2dot divided by 18 */
				 double a1by6[3], /* a1dot divided by 6 */
				 double aby2[3], /* a divided by 2 */
				 double v[3], /* velocity */
				 double x[3] /* position */)
{
  return predict_vector_on_emulator(xp, vp, *ti, *tj, *dtj, *tunit, *xunit,
			     a2by18,  a1by6, aby2,  v, x);
}


#ifdef TEST
main()
{
    double tj, dtj, ti, x[5], xp[2];
    ULONG xunit, tunit, err;
    ULONG tjlsb, dtjmsb, ix[5], ipred[2], iti;
    ULONG ntjlsb, ndtjmsb, nix[5];
    double pred[2];
    int i, dtjexp;
    set_debug_level(0);
    while (1){
      int nread;
      printf("enter tj, dtj, ti, tunit, xunit, x[0..4]: ");
      nread = scanf("%le%le%le%ld%ld%le%le%le%le%le",&tj, &dtj, &ti,&tunit, &xunit,
	    x,x+1,x+2,x+3,x+4);
      if(nread < 10) exit(0);
      err = convert_predictor(&tjlsb, &dtjmsb, ix, tj, dtj, x, xunit, tunit);
      printf("a, n = 0x%lx 0x%lx\n", tjlsb, dtjmsb);
      for(i=0;i<5;i++){
	printf("p[%1d]=%16lx %16lx %16lx\n",i, ix[i],nix[i],ix[i]-nix[i]);
      }
      frexp(dtj,&dtjexp);
      dtjexp = - dtjexp+1;
      for(i=0;i<2;i++){
	predict_double(pred+i, (ULONG)i, ti, tj, x[4],x[3],x[2],x[1],x[0]);
      }
      convert_predicted_result(xp, xp+1,ix[0], ix[1],  xunit, tunit);
      printf("converted x, v = %le %le \n", *xp, *(xp+1));
      
      
      iti = (ULONG) ldexp(ti, (int)tunit);
      for(i=0;i<2;i++){
	predict(ipred+i, (ULONG)i, iti, tjlsb, dtjmsb,
		ix[4], ix[3], ix[2], ix[1], ix[0]);
      }
      printf("IN= %16lx %8lx %8lx %8lx %8lx %8lx %8lx %16lx OUT= %16lx %8lx\n",
	     iti, tjlsb, dtjmsb,ix[4], ix[3], ix[2], ix[1], ix[0], ipred[0], ipred[1] );
      printf(" xp, vp = %lx %lx \n", ipred[0], ipred[1]);
      convert_predicted_result(xp, xp+1,ipred[0], ipred[1],  xunit, tunit);
      printf("predict_double result=%21.14le %21.14le\n",  pred[0],pred[1]);
      printf("converted xp, vp    = %21.14le %21.14le \n", *xp, *(xp+1));
      printf("rel err             = %21.14le %21.14le \n", (*xp-pred[0])/pred[0],(xp[1]-pred[1])/pred[1]);
      printf("vps in hex: %lx %lx\n", *((long*)(xp+1)), *((long*)(pred+1)));
    }
}
#endif



#ifdef HEXTEST
main()
{
    ULONG xunit, tunit, err;
    ULONG tjlsb, dtjmsb, ix[5], ipred[2], iti;
    ULONG ntjlsb, ndtjmsb, nix[5];
    ULONG passk4, passk4a, passk3a, passk2a, passk1a, passk3d,
	passk3e, passk2c, passk2d;

    double xp[2];
    int i, dtjexp;
    set_debug_level(0);
    printf("enter  tunit, xunit ");
    scanf("%ld%ld",&tunit, &xunit);
    while (1){
	int nread;
	nread = scanf("%lx%lx%lx%lx%lx%lx%lx%lx%lx%lx%lx%lx%lx%lx%lx%lx%lx",
		      &tjlsb, &dtjmsb, &iti, ix+4, ix+3, ix+2, ix+1, ix,
		      &passk4, &passk4a, &passk3a, &passk2a, &passk1a, &passk3d,
		      &passk3e, &passk2c, &passk2d);
	
	if(nread < 17) exit(0);
	for(i=0;i<2;i++){
	    predict_with_testmode(ipred+i, (ULONG)i, iti, tjlsb, dtjmsb,
				  ix[4], ix[3], ix[2], ix[1], ix[0],
				  passk4, passk4a, passk3a, passk2a, passk1a,
				  passk3d,passk3e, passk2c, passk2d);
	}
	printf("IN= %16lx %8lx %8lx %8lx %8lx %8lx %8lx %16lx",
	       iti, tjlsb, dtjmsb,ix[4], ix[3], ix[2], ix[1], ix[0]);
	printf(" %1lx %1lx %1lx %1lx %1lx %1lx %1lx %1lx %1lx",
	       passk4, passk4a, passk3a, passk2a, passk1a,
	       passk3d,passk3e, passk2c, passk2d);
	printf(" OUT= %16lx %8lx\n",ipred[0], ipred[1] );
      printf(" xp, vp = %lx %lx \n", ipred[0], ipred[1]);
      convert_predicted_result(xp, xp+1,ipred[0], ipred[1],  xunit, tunit);
      printf("converted xp, vp    = %21.14le %21.14le \n", *xp, *(xp+1));
    }
}
#endif


